import ContextMenu from '../kendocontextmenu';
import Menu from '../menu';

const SUB_MENU_CLASS = 'cv-k-appended-sub-menu';

function _buildItemMap(menuName, options, map, parentId) {
	if (!map) {
		map = {};
	}
	if (!_.isArray(options) || options.length === 0) {
		return map;
	}
	_.forEach(options, option => {
		let optionObj = map[option.id];
		if (!optionObj) {
			optionObj = {
				id: option.id,
				visible: !option.hidden,
				parentId: parentId,
				childrenIds: []
			};
			if (!parentId) {
				// If root-level, add a context menu
				optionObj.contextMenu = new ContextMenu({
					options: option.subOptions || [],
					filter: `[data-cv-menu-item-id="${option.id}"]`,
					cssClass: SUB_MENU_CLASS,
					showOn: 'click',
					name: `${menuName}_${option.id}_contextMenu`
				});
				if (!_.isArray(option.subOptions) || option.subOptions.length === 0) {
					optionObj.contextMenu.element.parent('.k-animation-container').addClass('hidden');
				}
			}
			map[option.id] = optionObj;
		}
		if (parentId) {
			map[parentId].childrenIds.push(option.id);
		}
		_buildItemMap(menuName, option.subOptions, map, option.id);
	});
	return map;
}

function _removeFromItemMap(id, map) {
	const itemObj = map[id];
	if (itemObj) {
		if (itemObj.contextMenu) {
			itemObj.contextMenu.destroy();
		}
		delete map[id];

		const parentItemObj = map[itemObj.parentId];
		if (parentItemObj && parentItemObj.contextMenu) {
			parentItemObj.contextMenu.remove(id);
			parentItemObj.childrenIds = parentItemObj.childrenIds.filter(childId => childId !== id);
		}

		_.forEach(itemObj.childrenIds, childId => {
			_removeFromItemMap(childId, map);
		});
	}
}

function _getRootFromItemMap(id, map) {
	const itemObj = map[id];
	if (itemObj) {
		if (itemObj.parentId) {
			return _getRoot(itemObj.parentId, map);
		}
		return itemObj;
	}
	return null;
}

/**
 * Functionally the same as Menu, except the submenus are instead defined as
 * context menus that are appended to the body. Useful in cases where submenus
 * are being covered while using Menu.
 */
export default class AppendedMenu extends Menu {
	constructor(args) {
		super(args);
	}

	_buildOptionText(option) {		
		let optionText = super._buildOptionText(option);
		optionText = optionText.substring(0, optionText.lastIndexOf(`</a>`));
		const hasSubOptions = _.isArray(option.subOptions) && option.subOptions.length > 0;
		return (
			optionText +
			`
				<span class="k-icon k-i-arrow-60-down k-menu-expand-arrow ${hasSubOptions ? '' : 'hidden'}">
				</span>
			</a>
		`
		);
	}

	initialize(args) {
		if (args.element && !this.name) {
			const originalElementId = args.element.attr('id');
			if (originalElementId) {
				// If no name is specified and the menu is built on a predefined
				// element with an id, use the existing id as the menu name
				this.name = originalElementId;
			}
		}
		this._itemMap = _buildItemMap(this.name, args.options);
		const baseMenuOptions = _.map(args.options, option => ({
			...option,
			subOptions: null
		}));
		super.initialize({
			...args,
			options: baseMenuOptions
		});
	}

	build() {
		super.build();
		this._refreshMenuArrows();
	}

	_itemHasVisibleChildren(id) {
		const itemObj = this._itemMap[id];
		return _.some(itemObj.childrenIds, childId => this._itemMap[childId].visible);
	}

	_refreshMenuArrows() {
		_.forEach(this.optionsMap, (option, id) => {
			this._refreshMenuArrow(id);
		});
	}

	_refreshMenuArrow(id) {
		const itemObj = this._itemMap[id];
		if (itemObj && itemObj.contextMenu) {
			const arrowElement = this._getOptionElement(id).find('.k-menu-expand-arrow');
			const contextMenuElement = itemObj.contextMenu.element.parent('.k-animation-container');

			if (_.some(itemObj.childrenIds, childId => this._itemMap[childId].visible)) {
				arrowElement.removeClass('hidden');
				contextMenuElement.removeClass('hidden');
			} else {
				arrowElement.addClass('hidden');
				contextMenuElement.addClass('hidden');
			}
		}
	}

	setVisibility(id, visible) {
		super.setVisibility(id, visible);
		const itemObj = this._itemMap[id];
		if (itemObj) {
			itemObj.visible = visible;
			this._refreshMenuArrow(itemObj.parentId);
		}
	}

	isVisible(id) {
		let isVisible = super.isVisible(id);
		if (!isVisible) {
			// ID is not visible in the root menu, check if it is visible in a
			// context menu
			const rootItemObj = _getRootFromItemMap(id, this._itemMap);
			if (rootItemObj && rootItemObj.contextMenu) {
				isVisible = rootItemObj.contextMenu.isVisible(id);
			}
		}
		return isVisible;
	}

	add(options, parentId) {
		if (!(parentId in this._itemMap)) {
			parentId = null;
		}
		_buildItemMap(this.name, options, this._itemMap, parentId);
		if (!parentId) {
			const baseMenuOptions = _.map(options, option => ({
				...option,
				subOptions: null
			}));
			super.add(baseMenuOptions);
		} else {
			// Add options to correct context menu
			let parentItemObj = this._itemMap[parentId];
			while (parentItemObj.parentId) {
				parentItemObj = this._itemMap[parentItemObj.parentId];
			}
			parentItemObj.contextMenu.add(options, parentItemObj.id);
			this._refreshMenuArrow(parentItemObj.id);
		}
	}

	removeAll() {
		super.removeAll();
		this._destroyAllContextMenus();
		this._itemMap = {};
	}

	remove(id) {
		super.remove(id);
		const itemObj = this._itemMap[id];
		if (itemObj) {
			_removeFromItemMap(id, this._itemMap);
			const parentItemObj = this._itemMap[itemObj.parentId];
			if (parentItemObj) {
				this._refreshMenuArrow(parentItemObj.id);
			}
		}
	}

	_destroyAllContextMenus() {
		_.forEach(this._itemMap, itemObj => {
			if (itemObj.contextMenu) {
				itemObj.contextMenu.destroy();
			}
		});
	}

	destroy() {
		super.destroy();
		this._destroyAllContextMenus();
	}
}
