import { appDesignerModule } from 'common/js/modules';

import { objToString } from './../util';

import {
	APPS_API_ENDPOINT,
	DEFAULT_HEADERS,
	DEFAULT_JSON_HEADERS,
	DEFAULT_VIEW,
	USER_GROUPS_API_ENDPOINT
} from '../constants';

var endpoint = APPS_API_ENDPOINT;

// (GET) /tables/{{tableId}}/views/{{viewId}}          - gets the table view
// (PUT) /tables/{{tableId}}/views/{{viewId}}          - updated the table view
// (GET) /tables/{{tableId}}/views                     - gets the list of views for a table
// (POST) /tables/{{tableId}}/views                    - creates a view for a table
// (DELETE) /tables/{{tableId}}/views/{{viewId}}       - deletes the view for a table

function ViewsService($http, $log, $q) {
	let headers = DEFAULT_HEADERS;
	let jsonHeaders = DEFAULT_JSON_HEADERS;
	let transformRequest = objToString;

	/**
	 * Views class which is returned as the base of the ViewsService service
	 */
	class Views {
		/**
		 * creates a instance of Views
		 * @param {string} [baseUrl] - optional base url to use for REST calls
		 */
		constructor(baseUrl) {
			this._baseUrl = baseUrl !== undefined ? baseUrl : `${APPS_API_ENDPOINT}/tables`;
		}

		/**
		 * @type {string}
		 */
		get baseUrl() {
			return this._baseUrl;
		}

		/**
		 * method to get all available tables
		 * @returns {Promise} $http promise
		 */
		getViews(tableId) {
			let endpoint = `${this.baseUrl}/${tableId}/views`;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
					// return instances of View
					.then(views => views.map(view => new View(tableId, view.sys_id, { view })))
			);
		}

		// /**
		//  * get table by ID
		//  * @param {string} - id of table
		//  * @returns {Promise} $http promise
		//  */
		// getViewById(id) {
		//   if (id === undefined) {
		//     throw new Error('id required');
		//   }
		//   return new View(id).promise;
		// }

		createView(table) {
			if (table === undefined) {
				return new View();
			} else {
				let endpoint = this.baseUrl;
				return $http
					.post(endpoint, table, { headers, transformRequest })
					.then(onRequestSuccess, onRequestFailure)
					.then(table => new View(table.sys_id, { table }));
			}
		}

		getViewFromData(tableId, id, view) {
			return new View(tableId, id, { view });
		}

		getInputForm(tableId) {
			let endpoint = this.baseUrl + `/${tableId}/form`;

			return $http.get(endpoint, { headers }).then(response => response.data);
		}

		getViewById(tableId, id) {
			if (tableId === undefined) {
				throw new Error('tableId required');
			}
			if (id === undefined) {
				throw new Error('id required');
			}
			let deferred = $q.defer();

			const getView = () =>
				new View(tableId, id).promise.then(
					view => {
						deferred.resolve(view);
					},
					err => {
						deferred.reject(err);
					}
				);

			if (id) {
				getView();
				return deferred.promise;
			}

			return this.getInputForm(tableId).then(res => new View(tableId, null, { view: { entries: res.entries } }));
		}

		getUserGroups() {
			return $http.get(USER_GROUPS_API_ENDPOINT, { jsonHeaders }).then(response => response.data);
		}
	}

	class View {
		/**
		 * creates a instance of View
		 * @param {string} tableId - id for the View instance
		 * @param {string} [id] - optional id for the View instance
		 * @param {object} [config] - configuration object containing baseUrl and column properties
		 */
		constructor(tableId, id, config = {}) {
			if (tableId === undefined) {
				throw new Error('table id must be set to create view');
			}
			// destructuring of config object
			let { baseUrl, view = DEFAULT_VIEW } = config;
			// promise which can be used
			this._deferred = $q.defer();
			// REST endpoint for app
			this._baseUrl = baseUrl;
			this.tableId = tableId;
			this.id = id;
			this.promise = this._deferred.promise;
			this.viewId = view.sys_rowId;
			this.resolved = false;
			// if id is supplied and app is set to DEFAULT_APP we need to fetch app details
			if (this.id && Object.is(view, DEFAULT_VIEW)) {
				this._getView();
			} else {
				this._setProperties(view);
				this.resolved = true;
			}
		}

		get baseUrl() {
			let baseUrl = this._baseUrl !== undefined ? this._baseUrl : `${APPS_API_ENDPOINT}/tables/${this.tableId}/views`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * view data model for creation and updates
		 * @private
		 */
		get _view() {
			return {
				title: this.name,
				description: this.description,
				view: this.view
			};
		}

		/**
		 * method to setup view model properties
		 * @private
		 */
		_setProperties(view) {
			if (!this.id) {
				this.id = view.sys_id || view.id;
			}

			this.name = view.title;
			this.description = view.description;
			this.view = view.view;
			this.entries = view.entries;

			this.version = view.sys_version || view.version;
			this.created = { at: view.sys_created_at || view.created };
			this.modified = Object.assign({ at: view.sys_created_at || view.created }, view.sys_modified || view.modified);

			return this;
		}

		/**
		 * method used by constructor to fetch column instance
		 * @private
		 */
		_getView() {
			let endpoint = `${this.baseUrl}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					column => {
						this._setProperties(column);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let promise;
			if (!this.id) {
				// create new
				let endpoint = `${this.baseUrl}`;
				promise = $http.post(endpoint, this._view, { headers, transformRequest });
			} else {
				// update existing
				let endpoint = `${this.baseUrl}`;
				promise = $http.put(endpoint, this._view, { headers, transformRequest });
			}
			return promise.then(onRequestSuccess, onRequestFailure).then(column => this._setProperties(column));
		}

		remove() {
			let endpoint = `${this.baseUrl}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}
	}

	function onRequestSuccess(res) {
		// basic transform to simplify service response handling
		return res.data;
	}

	function onRequestFailure(err) {
		// TODO: [app-designer] better error handler
		$log.error(err);
		return $q.reject(err);
	}

	return new Views();
}

ViewsService.$inject = ['$http', '$log', '$q'];

appDesignerModule.factory('ViewsService', ViewsService);

export default appDesignerModule;
