import { appDesignerModule } from 'common/js/modules';

import { cellNameTemplate, cellActionTemplate, objToString } from './../util';

import { fieldTypes } from '../table-fields/constants';

import 'users/js/filters/users.filters.js';
import 'adminConsole/js/filters/clientGroups.filters.js';
import 'machineBrowse/js/controllers/machineBrowse.ctrl.js';
import 'adminConsole/js/controllers/clientGroups.ctrl.js';
import 'vsa/js/controllers/collections.ctrl.js';
import 'modules/servers/js/services/servers.svc.js';
import 'modules/ida/js/services/idas.factory.js';
import 'adminConsole/js/controllers/security.ctrl.js';
import 'adminConsole/js/controllers/subclients.ctrl.js';
import 'storage/js/services/storage.svc.js';
import 'dlo/js/services/profile.svc.js';

import {
	APPS_API_ENDPOINT,
	FILE_STORES_API_ENDPOINT,
	DEFAULT_HEADERS,
	DEFAULT_TABLE,
	DEFAULT_COLUMN,
	DEFAULT_COLABORATION_SHARE
} from '../constants';
import { pathAccess } from '../util';

var endpoint = APPS_API_ENDPOINT;

function TablesService($http, $log, $q) {
	let headers = DEFAULT_HEADERS;
	let transformRequest = objToString;

	/**
	 * Tables class which is returned as the base of the TablesService service
	 */
	class Tables {
		/**
		 * creates a instance of Tables
		 * @param {string} [baseUrl] - optional base url to use for REST calls
		 */
		constructor(baseUrl) {
			this._baseUrl = baseUrl !== undefined ? baseUrl : `${APPS_API_ENDPOINT}/tables`;
		}

		/**
		 * @type {string}
		 */
		get baseUrl() {
			return this._baseUrl;
		}

		/**
		 * method to get all available tables
		 * @returns {Promise} $http promise
		 */
		getTables() {
			let endpoint = this.baseUrl;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
					// return instances of Table
					.then(tables => tables.map(table => new Table(table.sys_id, { table })))
			);
		}

		/**
		 * get table by ID
		 * @param {string} - id of table
		 * @returns {Promise} $http promise
		 */
		getTableById(id) {
			if (id === undefined) {
				throw new Error('id required');
			}
			return new Table(id).promise;
		}

		createTable(table) {
			if (table === undefined) {
				return new Table();
			} else {
				let endpoint = this.baseUrl;
				return $http
					.post(endpoint, table, { headers, transformRequest })
					.then(onRequestSuccess, onRequestFailure)
					.then(table => new Table(table.sys_id, { table }));
			}
		}

		getColumnById(tableId, id) {
			if (tableId === undefined) {
				throw new Error('tableId required');
			}
			if (id === undefined) {
				throw new Error('id required');
			}
			let deferred = $q.defer();
			new Column(tableId, id).promise.then(
				column => {
					deferred.resolve(column);
				},
				err => {
					deferred.reject(err);
				}
			);

			return deferred.promise;
		}

		getColumnTypes() {
			// https://cvs.commvault.com/source/xref/11.0-win/vaultcx/Source/CommClient/CustomReportsEngine/src/commvault/web/reportsplus/type/ReportDataType.java#18-31
			return [
				'String',
				'Integer',
				'Long',
				'Boolean',
				'Date',
				'Time',
				'TimeStamp',
				'Float',
				'Double',
				'Short',
				'Decimal',
				'DateRange'
			];
		}

		getFieldTypes() {
			return fieldTypes;
		}

		getFileStores() {
			const headers = DEFAULT_COLABORATION_SHARE;
			const endpoint = FILE_STORES_API_ENDPOINT;

			return $http.get(endpoint, { headers, transformRequest }).then(resp => resp.data.fileStores);
		}
	}

	// The classes below are not directly exposed by the service but are support of the Tables class

	/**
	 * Table class used for table instances within the Tables service
	 */
	class Table {
		/**
		 * creates a instance of Table
		 * @param {string} [id] - optional id for the Table instance
		 * @param {object} [config] - configuration object containing baseUrl and table properties
		 */
		constructor(id, config = {}) {
			// destructuring of config object
			let { baseUrl, table = DEFAULT_TABLE } = config;

			// promise which can be used
			this._deferred = $q.defer();

			// REST endpoint for table
			this._baseUrl = baseUrl;

			// columns
			this._columns = false;

			this.id = id;
			this.promise = this._deferred.promise;
			this.resolved = false;

			// if id is supplied and table is set to DEFAULT_TABLE we need to fetch table details
			if (this.id !== undefined && Object.is(table, DEFAULT_TABLE)) {
				this._getTable();
			} else {
				this._setProperties(table);
				this.resolved = true;
			}
		}

		get columns() {
			if (this._columns === false) {
				this._columns = [];
				this.getColumns().then(columns => (this._columns = columns));
			}
			return this._columns;
		}

		get baseUrl() {
			let baseUrl = this._baseUrl !== undefined ? this._baseUrl : `${APPS_API_ENDPOINT}/tables`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * table data model for creation and updates
		 * @private
		 */
		get _table() {
			return {
				name: this.name,
				description: this.description
			};
		}

		/**
		 * method to setup view model properties
		 * @private
		 */
		_setProperties(table) {
			if (table.sys_id) {
				this.id = table.sys_id;
			}
			(this.rowId = table.sys_rowId), (this.name = table.name);
			this.description = table.description;
			this.created = table.sys_created;
			this.modified = Object.assign({ at: table.sys_created_at }, table.sys_modified);
			this.version = table.sys_version;
			this.domain = table.sys_domainId;
			this.totalRecords = table.totalRecords;

			return this;
		}

		/**
		 * method used by constructor to fetch app instance
		 * @private
		 */
		_getTable() {
			let endpoint = `${this.baseUrl}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					table => {
						this._setProperties(table);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let endpoint = `${this.baseUrl}`;
			let promise;
			if (this.id === undefined) {
				// create new
				promise = $http.post(endpoint, this._table, { headers, transformRequest });
			} else {
				// update existing
				promise = $http.put(endpoint, this._table, { headers, transformRequest });
			}

			return promise.then(onRequestSuccess, onRequestFailure).then(table => this._setProperties(table));
		}

		remove() {
			let endpoint = `${this.baseUrl}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}

		createColumn(column) {
			if (column === undefined) {
				return new Column(this.id);
			} else {
				let endpoint = `${this.baseUrl}/columns`;
				return $http
					.post(endpoint, column, { headers, transformRequest })
					.then(onRequestSuccess, onRequestFailure)
					.then(column => new Column(this.id, column.sys_id, { column }));
			}
		}

		/**
		 * method to get all available columns for the table
		 * @returns {Promise} $http promise
		 */
		getColumns() {
			let endpoint = `${this.baseUrl}/columns`;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
					// return instances of Column
					.then(columns => columns.map(column => new Column(this.id, column.sys_id, { column })))
					.then(columns => {
						this._columns = columns;
						return columns;
					})
			);
		}
	}

	class Column {
		/**
		 * creates a instance of Column
		 * @param {string} tableId - id for the Table instance
		 * @param {string} [id] - optional id for the Column instance
		 * @param {object} [config] - configuration object containing baseUrl and column properties
		 */
		constructor(tableId, id, config = {}) {
			if (tableId === undefined) {
				throw new Error('table id must be set to create column');
			}
			// destructuring of config object
			let { baseUrl, tablesBaseUrl, column = DEFAULT_COLUMN } = config;

			// promise which can be used
			this._deferred = $q.defer();

			// REST endpoint for app
			this._baseUrl = baseUrl;
			this._tablesBaseUrl = tablesBaseUrl;

			this.tableId = tableId;

			this.id = id;
			this.promise = this._deferred.promise;
			this.resolved = false;
			// if id is supplied and app is set to DEFAULT_APP we need to fetch app details
			if (this.id !== undefined && Object.is(column, DEFAULT_COLUMN)) {
				this._getColumn();
			} else {
				this._setProperties(column);
				this.resolved = true;
			}
		}

		get tablesBaseUrl() {
			// use for create or get
			return this._tablesBaseUrl !== undefined
				? this._tablesBaseUrl
				: `${APPS_API_ENDPOINT}/tables/${this.tableId}/columns`;
		}

		get baseUrl() {
			let baseUrl = this._baseUrl !== undefined ? this._baseUrl : `${APPS_API_ENDPOINT}/columns`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * column data model for creation and updates
		 * @private
		 */
		get _column() {
			const dataSetEntity = pathAccess(['choices', 'dataSetEntity'], this.fieldOptions);

			if (dataSetEntity) delete this.fieldOptions.choices.dataSetEntity['$$hashKey'];
			if (this.type !== 'File' && this.fieldOptions.shareId) delete this.fieldOptions.shareId;

			return {
				name: this.name,
				displayName: this.displayName,
				defaultValue: this.defaultValue,
				description: this.description,
				type: this.type,
				fieldOptions: this.fieldOptions || {},
				unique: this.unique,
				required: this.required,
				isList: this.isList
			};
		}

		/**
		 * method to setup view model properties
		 * @private
		 */
		_setProperties(column) {
			this.id = column.sys_id;
			this.name = column.name;
			this.displayName = column.displayName;
			this.description = column.description;
			this.type = column.type;
			this.unique = column.unique;
			this.required = column.required;
			this.isList = column.isList;
			this.defaultValue = column.defaultValue;
			this.fieldOptions = column.fieldOptions || {};

			this.version = column.sys_version;
			this.created = { at: column.sys_created_at };
			this.modified = Object.assign({ at: column.sys_created_at }, column.sys_modified);

			return this;
		}

		/**
		 * method used by constructor to fetch column instance
		 * @private
		 */
		_getColumn() {
			let endpoint = `${this.baseUrl}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					column => {
						this._setProperties(column);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let promise;
			if (this.id === undefined) {
				// create new
				let endpoint = `${this.tablesBaseUrl}`;
				promise = $http.post(endpoint, this._column, { headers, transformRequest });
			} else {
				// update existing
				let endpoint = `${this.baseUrl}`;
				promise = $http.put(endpoint, this._column, { headers, transformRequest });
			}
			return promise.then(onRequestSuccess, onRequestFailure).then(column => this._setProperties(column));
		}

		remove() {
			let endpoint = `${this.baseUrl}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}
	}

	function onRequestSuccess(res) {
		// basic transform to simplify service response handling
		return res.data;
	}

	function onRequestFailure(err) {
		// TODO: [app-designer] better error handler
		$log.error(err);
		return $q.reject(err);
	}

	return new Tables();
}

TablesService.$inject = ['$http', '$log', '$q'];

appDesignerModule.factory('TablesService', TablesService);

export default appDesignerModule;
