import { appDesignerModule } from 'common/js/modules';

import { objToString } from './../util';

import { APPS_API_ENDPOINT, VIEWER_API_ENDPOINT, DEFAULT_HEADERS, DEFAULT_APP, DEFAULT_PAGE } from '../constants';

import '../services/business.svc';
import '../services/appForms.svc';
import '../services/views.svc';
import '../services/tables.svc';
import '../services/reports.svc';

import capitalize from 'lodash/capitalize';

function AppsService($http, $log, $q, cvLoc, ViewsService, TablesService, ReportsService) {
	let headers = DEFAULT_HEADERS;
	let transformRequest = objToString;

	/**
	 * Apps class which is returned as the base of the
	 * AppsService service
	 */
	class Apps {
		/**
		 * creates a instance of Apps
		 * @param {string} [baseUrl] - optional base url to use for REST calls
		 */
		constructor(baseUrl) {
			this._baseUrl = baseUrl !== undefined ? baseUrl : `${APPS_API_ENDPOINT}/apps`;
		}

		/**
		 * @type {string}
		 */
		get baseUrl() {
			return this._baseUrl;
		}

		/**
		 * method to get all available apps
		 * @returns {Promise} $http promise
		 */
		getApps() {
			let endpoint = this.baseUrl;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
					// return instances of App
					.then(apps => apps.map(app => new App(app.sys_id, { app })))
			);
		}

		/**
		 * get app by ID
		 * @param {string} - id of app
		 * @returns {Promise} $http promise
		 */
		getAppById(id) {
			if (id === undefined) {
				throw new Error('id required');
			}
			return new App(id).promise;
		}

		createApp(app) {
			if (app === undefined) {
				return new App();
			} else {
				let endpoint = this.baseUrl;
				return $http
					.post(endpoint, app, { headers, transformRequest })
					.then(onRequestSuccess, onRequestFailure)
					.then(app => new App(app.sys_id, { app }));
			}
		}

		getPageById(appId, id) {
			if (appId === undefined) {
				throw new Error('appId required');
			}
			if (id === undefined) {
				throw new Error('id required');
			}
			let deferred = $q.defer();
			new Page(appId, id).promise.then(
				page => {
					deferred.resolve(page);
				},
				err => {
					deferred.reject(err);
				}
			);

			return deferred.promise;
		}

		getMenu(id) {
			return getViewerMenu(id);
		}

		getPageTypes() {
			return [
				{
					id: 'report',
					name: cvLoc('appdesigner.label.report'),
					getList: () => ReportsService.getReports()
				},
				{
					// this is for parent menu items
					id: 'table',
					name: cvLoc('appdesigner.label.table'),
					getList: () => TablesService.getTables(),
					views: {
						id: 'view',
						name: cvLoc('appdesigner.label.view'),
						getList: tableId => ViewsService.getViews(tableId)
					}
				}
			];
		}

		getPageTypeNameById(id) {
			return this.getPageTypes()
				.filter(pageType => id === pageType.id)
				.pop();
		}
	}

	// The classes below are not directly exposed by the service but are support of the Apps class

	/**
	 * App class used for app instances within the Apps service
	 */
	class App {
		/**
		 * creates a instance of App
		 * @param {string} [id] - optional id for the App instance
		 * @param {object} [config] - configuration object containing baseUrl and page properties
		 */
		constructor(id, config = {}) {
			// destructuring of config object
			let { baseUrl, app = DEFAULT_APP } = config;

			// promise which can be used
			this._deferred = $q.defer();

			// REST endpoint for app
			this._baseUrl = baseUrl;

			// pages
			this._pages = false;

			this.id = id;
			this.promise = this._deferred.promise;
			this.resolved = false;

			// if id is supplied and app is set to DEFAULT_APP we need to fetch app details
			if (this.id !== undefined && Object.is(app, DEFAULT_APP)) {
				this._getApp();
			} else {
				this._setProperties(app);
				this.resolved = true;
			}
		}

		get pages() {
			if (this._pages === false) {
				this._pages = [];
				this.getPages().then(pages => (this._pages = pages));
			}
			return this._pages;
		}

		get baseUrl() {
			let baseUrl = this._baseUrl !== undefined ? this._baseUrl : `${APPS_API_ENDPOINT}/apps`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * app data model for creation and updates
		 * @private
		 */
		get _app() {
			return {
				name: this.name,
				description: this.description,
				logo: this.logo
			};
		}

		/**
		 * method to setup view model properties
		 * @private
		 */
		_setProperties(app) {
			if (app.sys_id) {
				this.id = app.sys_id;
			}
			(this.rowId = app.sys_rowId), (this.name = app.name);
			this.description = app.description;
			this.logo = app.logo;
			this.created = app.sys_created;
			this.modified = Object.assign({ at: app.sys_created_at }, app.sys_modified);
			this.version = app.sys_version;

			return this;
		}

		/**
		 * method used by constructor to fetch app instance
		 * @private
		 */
		_getApp() {
			let endpoint = `${this.baseUrl}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					app => {
						this._setProperties(app);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let endpoint = `${this.baseUrl}`;
			let promise;

			if (this.id === undefined) {
				// create new
				promise = $http.post(endpoint, this._app, { headers, transformRequest });
			} else {
				// update existing
				promise = $http.put(endpoint, this._app, { headers, transformRequest });
			}
			return promise.then(onRequestSuccess, onRequestFailure).then(app => this._setProperties(app));
		}

		remove() {
			let endpoint = `${this.baseUrl}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}

		createPage(page) {
			if (page === undefined) {
				return new Page(this.id);
			} else {
				let endpoint = `${this.baseUrl}/menu`;
				return $http
					.post(endpoint, page, { headers, transformRequest })
					.then(onRequestSuccess, onRequestFailure)
					.then(page => new Page(this.id, page.sys_id, { page }));
			}
		}

		/**
		 * method to get all available pages for the app
		 * @returns {Promise} $http promise
		 */
		getPages() {
			let endpoint = `${this.baseUrl}/menu`;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
					// return instances of App
					.then(pages => pages.map(page => new Page(this.id, page.sys_id, { page })))
					.then(pages => {
						this._pages = pages;
						return pages;
					})
			);
		}

		getMenu() {
			return getViewerMenu(this.id);
		}
	}

	class Page {
		/**
		 * creates a instance of Page
		 * @param {string} appId - id for the App instance
		 * @param {string} [id] - optional id for the Page instance
		 * @param {object} [config] - configuration object containing baseUrl and page properties
		 */
		constructor(appId, id, config = {}) {
			if (appId === undefined) {
				throw new Error('app id must be set to create page');
			}
			// destructuring of config object
			let { baseUrl, page = DEFAULT_PAGE } = config;

			// promise which can be used
			this._deferred = $q.defer();

			// REST endpoint for app
			this._baseUrl = baseUrl;

			this.appId = appId;

			this.id = id;
			this.promise = this._deferred.promise;
			this.resolved = false;

			// if id is supplied and app is set to DEFAULT_APP we need to fetch app details
			if (this.id !== undefined && Object.is(page, DEFAULT_PAGE)) {
				this._getPage();
			} else {
				this._setProperties(page);
				this.resolved = true;
			}
		}

		get baseUrl() {
			let baseUrl = this._baseUrl !== undefined ? this._baseUrl : `${APPS_API_ENDPOINT}/apps/${this.appId}/menu`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * app data model for creation and updates
		 * @private
		 */
		get _page() {
			return {
				name: this.name,
				description: this.description,
				logo: this.logo,
				type: this.type,
				order: this.order,
				itemId: this.itemId,
				properties: this.properties
			};
		}

		/**
		 * method to setup view model properties
		 * @private
		 */
		_setProperties(page) {
			if (page.sys_id) {
				this.id = page.sys_id;
			}
			const pageType = new Apps().getPageTypeNameById(page.type);
			this.name = page.name;
			this.description = page.description;
			this.logo = page.logo;
			this.type = page.type;
			this.typeLabel = pageType ? pageType.name : capitalize(page.type);
			this.order = page.order;
			this.itemId = page.itemId;
			this.properties = page.properties || {};
			this.created = { at: page.sys_created_at };
			this.modified = Object.assign({ at: page.sys_created_at }, page.sys_modified);

			return this;
		}

		/**
		 * method used by constructor to fetch page instance
		 * @private
		 */
		_getPage() {
			let endpoint = `${this.baseUrl}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					page => {
						this._setProperties(page);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let endpoint = `${this.baseUrl}`;
			let promise;
			if (this.id === undefined) {
				// create new
				promise = $http.post(endpoint, this._page, { headers, transformRequest });
			} else {
				// update existing
				promise = $http.put(endpoint, this._page, { headers, transformRequest });
			}
			return promise.then(onRequestSuccess, onRequestFailure).then(page => this._setProperties(page));
		}

		remove() {
			let endpoint = `${this.baseUrl}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}
	}

	// TODO: [app-designer] I would prefer to have a clean view model here, not a full dump of data model
	function getViewerMenu(id) {
		let endpoint = `${VIEWER_API_ENDPOINT}/apps`;
		if (id !== undefined) {
			endpoint += `/${id}`;
		}
		return (
			$http
				.get(endpoint, { headers })
				// basic handlers
				.then(onRequestSuccess, onRequestFailure)
		);
	}

	function onRequestSuccess(res) {
		// basic transform to simplify service response handling
		return res.data;
	}

	function onRequestFailure(err) {
		// TODO: [app-designer] better error handler
		$log.error(err);
		return $q.reject(err);
	}

	return new Apps();
}

AppsService.$inject = ['$http', '$log', '$q', 'cvLoc', 'ViewsService', 'TablesService', 'ReportsService'];

appDesignerModule.factory('AppsService', AppsService);

export default appDesignerModule;
