import { appDesignerModule } from 'common/js/modules';

import { APPS_API_ENDPOINT, DEFAULT_HEADERS } from '../constants';

function AppFormsService($http, $log, $q) {
	let headers = DEFAULT_HEADERS;

	/**
	 * AppForms class which is returned as the base of the AppFormsService service
	 */
	class AppForms {
		/**
		 * creates a instance of AppForms
		 *
		 * @param {string}
		 *            [baseUrl] - optional base url to use for REST calls
		 */
		constructor(baseUrl) {
			this._baseUrl = baseUrl !== undefined ? baseUrl : `${APPS_API_ENDPOINT}`;
		}

		/**
		 * @type {string}
		 */
		getBaseUrl() {
			return this._baseUrl;
		}

		/**
		 * method to get all available forms
		 *
		 * @returns {Promise} $http promise
		 */
		getAppForms(tableId) {
			let endpoint = this._baseUrl + `/tables/${tableId}/forms`;
			return (
				$http
					.get(endpoint, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
			);
		}

		updateInputForm(tableId, formId, data) {
			let endpoint = this._baseUrl + `/tables/${tableId}/forms/${formId}`;
			return (
				$http
					.put(endpoint, data, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
			);
		}

		createInputForm(tableId, data) {
			let endpoint = this._baseUrl + `/tables/${tableId}/forms/`;
			return $http.post(endpoint, data, { headers }).then(onRequestSuccess, onRequestFailure);
		}

		delete(tableId, formId) {
			let endpoint = this._baseUrl + `/tables/${tableId}/forms/${formId}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}

		/**
		 * method to get inputForm
		 *
		 * @returns {Promise} $http promise
		 */
		getInputForm(tableId, formId) {
			let endpoint = `retrieveAppTableForm.do`;
			return (
				$http
					.post(endpoint, { tableId, formId }, { headers })
					// basic handlers
					.then(onRequestSuccess, onRequestFailure)
			);
		}
	}

	function onRequestSuccess(res) {
		return res.data;
	}

	function onRequestFailure(err) {
		$log.error(err);
		return $q.reject(err);
	}

	return new AppForms();
}

AppFormsService.$inject = ['$http', '$log', '$q'];

appDesignerModule.factory('AppFormsService', AppFormsService);

export default appDesignerModule;
