import { appDesignerModule } from 'common/js/modules';

import { objToString } from './../util';

import { APPS_API_ENDPOINT, DEFAULT_HEADERS, DEFAULT_VIEW, WORKFLOWS_API_ENDPOINT } from '../constants';

var endpoint = APPS_API_ENDPOINT;

// (GET) /tables/{{tableId}}/views/{{viewId}}          - gets the table view
// (PUT) /tables/{{tableId}}/views/{{viewId}}          - updated the table view
// (GET) /tables/{{tableId}}/views                     - gets the list of views for a table
// (POST) /tables/{{tableId}}/views                    - creates a view for a table
// (DELETE) /tables/{{tableId}}/views/{{viewId}}       - deletes the view for a table

function ActionsService($http, $log, $q) {
	let headers = DEFAULT_HEADERS;
	let transformRequest = objToString;

	/**
	 * Views class which is returned as the base of the ViewsService service
	 */
	class Actions {
		/**
		 * creates a instance of Views
		 *
		 * @param {string}
		 *            [baseUrl] - optional base url to use for REST calls
		 */
		constructor(baseUrl) {
			this._baseUrl = baseUrl !== undefined ? baseUrl : `${APPS_API_ENDPOINT}/tables`;
		}

		/**
		 * @type {string}
		 */
		get baseUrl() {
			return this._baseUrl;
		}

		/**
		 * method to get all available tables
		 *
		 * @returns {Promise} $http promise
		 */
		getActions(tableId, returnOrignalResp) {
			let endpoint = `${this.baseUrl}/${tableId}/actions`;
			return returnOrignalResp != undefined && returnOrignalResp === true
				? $http
						.get(endpoint, { headers })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
				: $http
						.get(endpoint, { headers })
						// basic handlers
						.then(onRequestSuccess, onRequestFailure)
						// return instances of View
						.then(actions => actions.map(action => new Action(tableId, action)));
		}

		getActionFromData(tableId, action) {
			return new Action(tableId, action);
		}

		getActionById(tableId, id) {
			if (tableId === undefined) {
				throw new Error('tableId required');
			}
			if (id === undefined) {
				throw new Error('id required');
			}
			let deferred = $q.defer();

			const getAction = () =>
				new Action(tableId, { id: id }, true).promise.then(
					action => {
						deferred.resolve(action);
					},
					err => {
						deferred.reject(err);
					}
				);

			if (id) {
				getAction();
				return deferred.promise;
			}

			return this.getInputForm(tableId).then(res => new View(tableId, null, { view: { entries: res.entries } }));
		}

		getForms() {
			return $http.get(`${WORKFLOWS_API_ENDPOINT}`).then(onRequestSuccess, onRequestFailure);
		}
	}

	class Action {
		/**
		 * creates a instance of View
		 *
		 * @param {string}
		 *            tableId - id for the View instance
		 * @param {string}
		 *            [id] - optional id for the View instance
		 * @param {object}
		 *            [config] - configuration object containing baseUrl and column properties
		 */
		constructor(tableId, action, fetchActionDetails) {
			// destructuring of action object
			this.id = action.id;
			this.name = action.name;
			this.type = action.clickExpression && action.clickExpression.action ? action.clickExpression.action : '';
			this.properties = action.properties;
			this.tableId = tableId;
			this._deferred = $q.defer();
			this.promise = this._deferred.promise;

			if (fetchActionDetails != undefined && fetchActionDetails === true) {
				this._getAction();
			} else {
				this._setProperties(action);
				this.resolved = true;
			}
		}

		get baseUrl() {
			let baseUrl = this._baseUrl !== undefined ? this._baseUrl : `${APPS_API_ENDPOINT}/tables/${this.tableId}/actions`;
			return this.id === undefined ? baseUrl : `${baseUrl}/${this.id}`;
		}

		/**
		 * view data model for creation and updates
		 *
		 * @private
		 */
		get _action() {
			return {
				title: this.name,
				type: this.type,
				properties: this.properties,
				id: this.id
			};
		}

		/**
		 * method to setup view model properties
		 *
		 * @private
		 */
		_setProperties(action) {
			if (!this.id) {
				this.id = action.id;
			}

			this.name = action.name;
			this.type = action.type;
			this.properties = action.properties;

			return this;
		}

		/**
		 * method used by constructor to fetch column instance
		 *
		 * @private
		 */
		_getAction() {
			let endpoint = `${this.baseUrl}`;
			return $http
				.get(endpoint, { headers, transformRequest })
				.then(onRequestSuccess, onRequestFailure)
				.then(
					action => {
						this._setProperties(action);
						this._deferred.resolve(this);
						this.resolved = true;
						return this;
					},
					err => {
						this._deferred.reject(err);
					}
				);
		}

		save() {
			let promise;
			if (!this.id) {
				// create new
				let endpoint = `${this.baseUrl}`;
				promise = $http.post(endpoint, this._action, { headers, transformRequest });
			} else {
				// update existing
				let endpoint = `${this.baseUrl}`;
				promise = $http.put(endpoint, this._action, { headers, transformRequest });
			}
			return promise.then(onRequestSuccess, onRequestFailure).then(action => this._setProperties(action));
		}

		remove() {
			let endpoint = `${this.baseUrl}`;
			return $http.delete(endpoint, { headers }).then(onRequestSuccess, onRequestFailure);
		}
	}

	function onRequestSuccess(res) {
		// basic transform to simplify service response handling
		return res.data;
	}

	function onRequestFailure(err) {
		// TODO: [app-designer] better error handler
		$log.error(err);
		return $q.reject(err);
	}

	return new Actions();
}

ActionsService.$inject = ['$http', '$log', '$q'];

appDesignerModule.factory('ActionsService', ActionsService);

export default appDesignerModule;
