import { appDesignerModule } from 'common/js/modules';

// App Designer Utility
import { cellNameTemplate, cellActionTemplate } from './../util';

// temporary localization App Designer strings
import './../temp.localization';

// dependencies injected by Angular
import '../services/tables.svc';
import '../services/tableEditModal.svc';
import { getErrMessage } from '../util';

function TablesListCtrl(
	$log,
	cvLoc,
	$dialogs,
	cvTableOptions,
	TablesService,
	TableEditModalService,
	cvToaster,
	$state
) {
	let vm = this;

	// ideally the initial value should be in the route resolver
	vm.tablesList = [];
	fetchTables();

	vm.editTable = function(table = TablesService.createTable()) {
		var isNewTable = table.id ? false : true;

		function navigateToTableConfigScreen(currTable) {
			if (isNewTable == true) $state.go('tableConfig', { tableId: currTable.id });
			else fetchTables();
		}

		return TableEditModalService.openModal(table).then(navigateToTableConfigScreen);
	};

	vm.deleteTable = function(table) {
		let tableId = table.id;
		let tableName = table.name;

		$dialogs.confirm(
			cvLoc('appdesigner.label.confirmDelete'),
			cvLoc('appdesigner.msg.confirmDelete', `<b>${tableName}</b>`),
			{
				noFunction: function() {},
				yesFunction: function() {
					table.remove().then(onDeleteSuccess, handleError);
				}
			}
		);

		function onDeleteSuccess() {
			fetchTables();
			cvToaster.showSuccessMessage({
				message: cvLoc('appdesigner.msg.deleteSuccessful', `<b>${tableName}</b>`)
			});
		}
	};

	vm.gridOptions = {
		cvGridTitle: cvLoc('appdesigner.label.tables'),
		cvTableName: 'tablesListTable',
		cvSearchFields: ['name', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noTables'),
			links: [
				{
					label: cvLoc('appdesigner.label.addTable'),
					onclick: vm.editTable
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addTable'),
				onclick: vm.editTable,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'tablesList',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						sort: { direction: 'asc', priority: 0 },
						displayName: cvLoc('appdesigner.table.col.name'),
						enableHiding: false,
						cellTemplate: cellNameTemplate('row.entity.name', 'tableConfig({ tableId: row.entity.id })')
						// width : '20%'
					},
					{
						field: 'description',
						displayName: cvLoc('appdesigner.table.col.desc'),
						enableHiding: false,
						width: '55%'
					},
					{
						name: 'action',
						width: '5%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.edit'),
								action: 'grid.appScope.editTable(row.entity)'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteTable(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	function fetchTables() {
		let tables = TablesService.getTables();
		tables.then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(res) {
			vm.tablesList = res;
		}

		function onRequestFailure(err) {
			// TODO: [app-designer] error handling
			$log.error(err);
		}

		return tables;
	}

	function handleError(err) {
		if (err) {
			const message = getErrMessage(err);
			cvToaster.showErrorMessage({
				message
			});
		}
		$log.error(err);
	}
}

TablesListCtrl.$inject = [
	'$log',
	'cvLoc',
	'$dialogs',
	'cvTableOptions',
	'TablesService',
	'TableEditModalService',
	'cvToaster',
	'$state'
];

appDesignerModule.controller('TablesListCtrl', TablesListCtrl);

export default appDesignerModule;
