import { appDesignerModule } from 'common/js/modules';

// App Designer Utility
import { cellNameTemplate, cellActionTemplate } from './../util';

// temporary localization App Designer strings
import './../temp.localization';

// dependencies injected by Angular
import '../services/apps.svc';
import '../services/appEditModal.svc';
import { getErrMessage } from '../util';

function AppsListCtrl(cvLoc, cvTableOptions, AppsService, AppEditModalService, $dialogs, cvToaster) {
	let vm = this;

	// ideally the initial value should be in the route resolver
	vm.appList = [];
	fetchApps();

	vm.editApp = function(app = AppsService.createApp()) {
		return AppEditModalService.openModal(app).then(fetchApps);
	};

	vm.deleteApp = function(app) {
		let appId = app.id;
		let appName = app.name;

		$dialogs.confirm(
			cvLoc('appdesigner.label.confirmDelete'),
			cvLoc('appdesigner.msg.confirmDelete', `<b>${appName}</b>`),
			{
				noFunction: function() {},
				yesFunction: function() {
					app.remove().then(onDeleteSuccess, onDeleteFailure);
				}
			}
		);

		function onDeleteSuccess() {
			fetchApps();
			cvToaster.showSuccessMessage({
				message: cvLoc('appdesigner.msg.deleteSuccessful', `<b>${appName}</b>`)
			});
		}

		function onDeleteFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}
	};

	vm.gridOptions = {
		cvGridTitle: cvLoc('appdesigner.label.apps'),
		cvTableName: 'appsListTable',
		cvSearchFields: ['name', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noApps'),
			links: [
				{
					label: cvLoc('appdesigner.label.addApp'),
					onclick: vm.editApp
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addApp'),
				onclick: vm.editApp,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'appList',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						sort: { direction: 'asc', priority: 0 },
						displayName: cvLoc('appdesigner.table.col.name'),
						enableHiding: false,
						cellTemplate: cellNameTemplate('row.entity.name', 'appConfig({ appId: row.entity.id })'),
						width: '20%'
					},
					{
						field: 'description',
						displayName: cvLoc('appdesigner.table.col.desc'),
						enableHiding: false,
						width: '65%'
					},
					{
						name: 'action',
						width: '5%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.edit'),
								action: 'grid.appScope.editApp(row.entity)'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deleteApp(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	function fetchApps() {
		let apps = AppsService.getApps();

		apps.then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(res) {
			vm.appList = res;
		}

		function onRequestFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}

		return apps;
	}
}

AppsListCtrl.$inject = ['cvLoc', 'cvTableOptions', 'AppsService', 'AppEditModalService', '$dialogs', 'cvToaster'];

appDesignerModule.controller('AppsListCtrl', AppsListCtrl);

export default appDesignerModule;
