import { appDesignerModule } from 'common/js/modules';

// App Designer Utility
import { cellNameTemplate, cellActionTemplate, formatSecurityAssociations } from './../util';

// temporary localization App Designer strings
import './../temp.localization';

// dependencies injected/need by Angular
import '../services/apps.svc';
import '../services/appEditModal.svc';
import '../services/pageEditModal.svc';
import '../services/showSecurityModal.svc';

import 'adminConsole/js/services/security.svc.js';
import 'adminConsole/js/controllers/security.ctrl.js';

import { getErrMessage, transformLabel } from '../util';
import capitalize from 'lodash/capitalize';

function AppConfigCtrl(
	$stateParams,
	$state,
	$log,
	cvLoc,
	$dialogs,
	cvToaster,
	cvTableOptions,
	AppsService,
	AppEditModalService,
	PageEditModalService,
	ShowSecurityModalService,
	securityService,
	securityFactory
) {
	let vm = this;
	let appId = $stateParams.appId;

	vm.loading = true;
	vm.appId = appId;
	vm.app = {};

	vm.formattedSecurityAssociations;
	vm.associationsLoaded = false;
	vm.entityType = 179;
	vm.genericEntity = 131;
	vm.hideInheritedAssociations = true;

	vm.handleAppEdit = function handleAppEdit() {
		AppEditModalService.openModal(vm.app).then(fetchApp);
	};

	// ideally the initial value should be in the route resolver :-|
	fetchApp();
	vm.pages = [];

	vm.editPage = function(page = vm.app.createPage()) {
		return PageEditModalService.openModal(page).then(fetchPages);
	};

	vm.editSecurity = function editSecurity() {
		let entity = {
			entityType: 180,
			entityId: vm.app.rowId,
			entityName: vm.app.name,
			enityType: 'App'
		};
		return ShowSecurityModalService.openModal(entity).then(() => fetchSecurityAssociations(vm.app.rowId));
	};

	vm.deletePage = function(page) {
		let pageName = page.name;

		$dialogs.confirm(
			cvLoc('appdesigner.label.confirmDelete'),
			cvLoc('appdesigner.msg.confirmDelete', `<b>${pageName}</b>`),
			{
				noFunction: function() {},
				yesFunction: function() {
					page.remove().then(onDeleteSuccess, onDeleteFailure);
				}
			}
		);

		function onDeleteSuccess() {
			fetchPages();
			cvToaster.showSuccessMessage({
				message: cvLoc('appdesigner.msg.deleteSuccessful', '<b>' + pageName + '</b>')
			});
		}

		function onDeleteFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}
	};

	vm.viewPageConfig = function(page) {
		let params = { appId: vm.appId, pageId: page.id };
		$state.go('appPageConfig', params);
	};

	vm.viewPagePreview = function(page) {
		let params = { appId: vm.appId, pageId: page.id };
		$state.go('appPageViewer', params);
	};

	vm.gridOptions = {
		cvGridTitle: cvLoc('appdesigner.label.pages'),
		cvIsPageTitle: false,
		cvTableName: 'appPagesTable',
		cvSearchFields: ['name', 'description'],
		cvOnGridEmpty: {
			message: cvLoc('appdesigner.msg.noPages'),
			links: [
				{
					label: cvLoc('appdesigner.label.addPage'),
					onclick: vm.editPage
				}
			]
		},
		cvPageLinks: [
			{
				label: cvLoc('appdesigner.label.addPage'),
				onclick: vm.editPage,
				disableMultiCommcell: true
			}
		],
		cvAppScope: vm,
		gridOptions: Object.assign(
			{
				data: 'pages',
				showGridFooter: true,
				columnDefs: [
					{
						field: 'name',
						sort: { direction: 'asc', priority: 0 },
						displayName: cvLoc('appdesigner.table.col.name'),
						enableHiding: false,
						cellTemplate: cellNameTemplate(
							'row.entity.name',
							'appPageConfig({ appId: grid.appScope.appId, pageId: row.entity.id })'
						),
						width: '20%'
					},
					{
						field: 'description',
						displayName: cvLoc('appdesigner.table.col.desc'),
						enableHiding: false,
						width: '50%'
					},
					{
						field: 'typeLabel',
						displayName: cvLoc('appdesigner.table.col.type'),
						enableHiding: false,
						width: '10%'
					},
					{
						field: 'order',
						displayName: cvLoc('appdesigner.table.col.order'),
						enableHiding: false,
						width: '10%'
					},
					{
						name: 'action',
						width: '5%',
						displayName: cvLoc('appdesigner.table.col.actions'),
						enableHiding: false,
						enableFiltering: false,
						enableSorting: false,
						cellTemplate: cellActionTemplate([
							{
								label: cvLoc('appdesigner.label.edit'),
								action: 'grid.appScope.viewPageConfig(row.entity)'
							},
							{
								label: cvLoc('appdesigner.label.delete'),
								action: 'grid.appScope.deletePage(row.entity)'
							}
						])
					}
				]
			},
			cvTableOptions.commonNgGridOptions
		)
	};

	function fetchSecurityAssociations(entityId) {
		securityFactory.getSecurityAssociation('APPSTUDIO_APP', entityId).success(function(data) {
			vm.formattedSecurityAssociations = formatSecurityAssociations(data.securityAssociations);
			vm.associationsLoaded = true;
		});
	}

	function fetchApp() {
		let req = AppsService.getAppById(appId);
		req.then(onGetAppSuccess, onGetAppFailure);

		function onGetAppSuccess(app) {
			vm.loading = false;
			vm.app = app;
			fetchSecurityAssociations(app.rowId);
			fetchPages();
		}

		function onGetAppFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
		}

		return req;
	}

	function fetchPages() {
		let req = vm.app.getPages();

		req.then(onRequestSuccess, onRequestFailure);

		function onRequestSuccess(pages) {
			vm.pages = pages;
		}

		function onRequestFailure(err) {
			if (err) {
				const message = getErrMessage(err);
				cvToaster.showErrorMessage({
					message
				});
			}
			$log.error(err);
		}

		return req;
	}
}

AppConfigCtrl.$inject = [
	'$stateParams',
	'$state',
	'$log',
	'cvLoc',
	'$dialogs',
	'cvToaster',
	'cvTableOptions',
	'AppsService',
	'AppEditModalService',
	'PageEditModalService',
	'ShowSecurityModalService',
	'securityService',
	'securityFactory'
];

appDesignerModule.controller('AppConfigCtrl', AppConfigCtrl);

export default appDesignerModule;
