#!/bin/ksh
LogMessage()
{
	echo $1>>$LOGDIR
}

Usage()
{
	echo ""
	echo "Usage: $0 UserName Password Server OutDir [-AccurateCLA | -IgnoreCLA]"
	echo "UserName      TSM Server user name"
	echo "Password      TSM Server password"
	echo "Server        TSM Server name"
	echo "OutDir        Location for output files"
	echo "-AccurateCLA  (optional) To run an extensive CLA calculation"
	echo "-IgnoreCLA    (optional) To ignore CLA calculation"
	echo ""
	echo "Note:"
	echo "*  -AccurateCLA might take a very long time and could potentially"
	echo "             hang a larger TSM server"
	echo ""
}

if [[ "$1" = "" || "$2" = "" || "$3" = "" || "$4" = "" ]]; then
	Usage
	exit 1
fi

case "$5_" in
	-[Aa][Cc][Cc][Uu][Rr][Aa][Tt][Ee][Cc][Ll][Aa]_)
		CLA=accurate
		;;
	-[Ii][Gg][Nn][Oo][Rr][Ee][Cc][Ll][Aa]_)
		CLA=ignore
		;;
	_)
		CLA=aprox
		;;
	*)
		Usage
		exit 1
		;;
esac



#Define Variables
ID=$1
PASS=$2
SERVER=$3
OUTDIR=$4/EDC_DATA
USEROPTFILE=$OUTDIR/edc_useroptions.txt
LOGDIR=$OUTDIR/predeployment_log.txt

EULA=eula.txt

if [[ ! -f $EULA ]]; then
	echo "The end user license file does not exist. Please make sure that you copy the file $EULA"
	exit 1
fi

more $EULA
echo -n "Do you accept the terms and conditions(Y/N)?"
read eula_agreement

if [[ "$eula_agreement" != "Y" && "$eula_agreement" != "y" ]]; then
	echo "Exiting the program as the EULA was not accepted"
	exit 1
fi

echo "Creating output directory to store data"
if [[ -a $OUTDIR ]]; then
	echo "Given directory[$OUTDIR] already exists. Please delete or rename the existing directory"
	exit 1
fi

mkdir -p $OUTDIR
if [[ $? -ne 0 ]]; then
	echo "Failed to create output directory. Please check if the given directory exists"
	exit $?
fi

LogMessage "TSM SERVER CONFIGURATION"
LogMessage "SERVER :"$SERVER
LogMessage "OUTDIR :"$OUTDIR
LogMessage "USEROPTFILE :"$USEROPTFILE


echo "SERVER : "$SERVER >> $USEROPTFILE
echo "OUTDIR : "$OUTDIR >> $USEROPTFILE
echo "CLA OPTION : "$CLA >> $USEROPTFILE

set CLOUDVERSIONFILE=CloudScriptVersion.txt
head -1 edc_tsm.sh > $OUTDIR/$CLOUDVERSIONFILE

echo "Collecting TSM Server information"
echo $SERVER 1>$OUTDIR/edc_server.txt

LogMessage "Collecting TSM Version information"
echo "Collecting TSM Version information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -dateformat=1 'show config' | grep "Storage Management Server" 1>$OUTDIR/tsm_version.txt
set errcode=$?
if [[ $errcode -ne 0 ]]; then
	LogMessage "Failed to collect TSM Version information with error code :"$errcode
	echo "Failed to collect TSM Version information. Continuing with other commands..."
fi

LogMessage "Collecting TSM Version ifnormation through database"
echo "Collecting TSM Version information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -dateformat=1 -comma -dataonly=yes -outfile=$OUTDIR/edc_tsmconfig.txt "SELECT VERSION, RELEASE FROM STATUS"
set errcode=$?
if [[ $errcode -ne 0 ]]; then
	LogMessage "Failed to collect TSM Version information with error code :"$errcode
	echo "Failed to collect TSM Version information. Continuing with other commands..."
fi

LogMessage "Collecting client->schedule association information"
echo "Collecting Client->Schedule association information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_assoc.txt -dateformat=1 -comma 'query association * *'
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect client->schedule association with error code :"$?" Removing association output file"
	echo "Failed to run command" 	# if rc > 0 then echo error msg and quit
	rm $OUTDIR/edc_assoc.txt
fi

LogMessage "Collecting client information"
echo "Collecting Client information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_node.txt -dataonly=yes -dateformat=1 -comma query node format=detailed
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect client information. Bailing since this is mandatory. Exit code:"$errcode
	echo "Failed to run command" 	# if rc > 0 then echo error msg and quit
	exit 1
fi

LogMessage "Collecting schedule information"
echo "Collecting Schedule information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_sched.txt -dateformat=1 -comma query schedule format=detailed
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect schedule information. Removing schedule output file. Exit code:"$errcode
	echo "Failed to run command" 	# if rc > 0 then echo error msg and quit
	rm $OUTDIR/edc_sched.txt
fi

#if [ -a $OUTDIR/edc_node.txt ]; then
#	while read line ;do
#		NODE_NAME=`echo $line | cut -d, -f1`
#		echo "Collecting CLA numbers for node :" $NODE_NAME
#		dsmadmc -id=$ID -password=$PASS -server=$SERVER -dateformat=1 -comma -dataonly=yes -outfile=$OUTDIR/edc_cla_$NODE_NAME.txt "SELECT MAX(CC.FILE_SIZE) FROM CONTENTS CC WHERE CC.NODE_NAME='$NODE_NAME' AND ( CC.AGGREGATED='No' OR CC.AGGREGATED LIKE '1/%' ) GROUP BY CC.FILE_NAME"
#		sleep 1
#	done < $OUTDIR/edc_node.txt
#fi

LogMessage "Collecting job information"
echo "Collecting Job information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_jobs.txt -dateformat=1 -comma "SELECT SCHEDULE_NAME, START_TIME, END_TIME, SUCCESSFUL, BYTES, ENTITY FROM SUMMARY SS WHERE SS.ACTIVITY = 'BACKUP' AND (YEAR(current date) - YEAR(SS.START_TIME)) < 3 ORDER BY START_TIME"
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	if [[ -a $OUTDIR/edc_jobs.txt ]]; then
		rm $OUTDIR/edc_jobs.txt
	fi
	LogMessage "Failed to collect job information. Removed job file output. Command exit code:"$errcode
	echo "Failed to collect Job information" 	# if rc > 0 then echo error msg and quit
fi

LogMessage "Collecting standalone libraries"
echo "Collecting Standalone Libraries information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_standalonelibs.txt -dateformat=1 -comma "SELECT * FROM VOLUMES WHERE DEVCLASS_NAME IN ('FILE', 'DISK')"
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect standalone libraries. Exit code:"$errcode
	echo "Failed to Standalone Library information" 	# if rc > 0 then echo error msg and quit
fi

echo "Collecting Libraries information"
LogMessage "Collecting libraries information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_libraries.txt -dateformat=1 -comma "SELECT * FROM LIBRARIES"
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect libraries information. Exit code:"$errcode
	echo "Failed to collect Library information" 	# if rc > 0 then echo error msg and quit
fi

echo "Collecting Drives information"
LogMessage "Collecting drives information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_drives.txt -dateformat=1 -comma "SELECT * FROM DRIVES"
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	if [[ -a $OUTDIR/edc_drives.txt ]]; then
		rm $OUTDIR/edc_drives.txt
		LogMessage "Failed to collect drives information. Removed drives output file."
	fi
	echo "Failed to collect Drive information" 	# if rc > 0 then echo error msg and quit
	LogMessage "Failed to collect drives information. Command exit code :"$errcode
fi

echo "Collecting Media information"
LogMessage "Collecting media information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_media.txt -dateformat=1 -comma "SELECT VV.VOLUME_NAME, VV.DEVCLASS_NAME, VV.EST_CAPACITY_MB, LV.LIBRARY_NAME, VV.STGPOOL_NAME FROM VOLUMES VV, LIBVOLUMES LV  WHERE LV.VOLUME_NAME = VV.VOLUME_NAME"
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	echo "Failed collect Media information" 	# if rc > 0 then echo error msg and quit
	LogMessage "Failed to collect media information. Command exit code:"$errcode
fi

echo "Collecting Storage Policy information"
LogMessage "Collecting storage policy information"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/edc_storagepolicy.txt -dateformat=1 -comma "SELECT DISTINCT LV.LIBRARY_NAME, VV.STGPOOL_NAME FROM VOLUMES VV, LIBVOLUMES LV  WHERE LV.VOLUME_NAME = VV.VOLUME_NAME"
set errcode=$?
if [[ $errcode -ne 0 ]]; then 		# check return code passed to function
	echo "Failed collect Storage Policy information" 	# if rc > 0 then echo error msg and quit
	LogMessage "Failed to collect storage policy information. Command exit code:"$errcode
fi

LogMessage "Collecting filespace information for dataviewCX tool"
echo "Collecting filespace information for dataviewCX tool"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -dataonly=yes -outfile=$OUTDIR/filename.csv -dateformat=1 -comma "SELECT SUM(PCT_UTIL* CAPACITY/100) from FILESPACES"
if [[ $? -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect filespace information for dataviewCX tool"
	echo "Failed to collect filespace information for dataviewCX tool" 	# if rc > 0 then echo error msg and quit
fi

LogMessage "Collecting filespace information - node wise for dataviewCX tool"
echo "Collecting filespace information - node wise for dataviewCX tool"
dsmadmc -id=$ID -password=$PASS -server=$SERVER -outfile=$OUTDIR/tsmservername.csv -dateformat=1 -comma "SELECT NODE_NAME,SUM(PCT_UTIL * CAPACITY/100) AS \"TOTAL_CAPACITY(MB)\" FROM FILESPACES GROUP BY NODE_NAME ORDER BY NODE_NAME"
if [[ $? -ne 0 ]]; then 		# check return code passed to function
	LogMessage "Failed to collect filespace information - node wise for dataviewCX tool"
	echo "Failed to collect filespace information - node wise for dataviewCX tool" 	# if rc > 0 then echo error msg and quit
fi

echo "Processing Job information"
if [ -a $OUTDIR/edc_jobs.txt ]; then
	set -A LINES `wc -l $OUTDIR/edc_jobs.txt`
	TOP=`expr $LINES - 3`
	END=`expr $TOP - 4`
	head -${TOP} $OUTDIR/edc_jobs.txt | tail -${END} > $OUTDIR/edc_jobs_processed.txt
	rm $OUTDIR/edc_jobs.txt
	while read line ;do
		CLIENT_NAME=`echo $line | cut -d, -f6`
		echo $line >> "$OUTDIR/edc_jobs_${CLIENT_NAME}.txt"
	done < $OUTDIR/edc_jobs_processed.txt
fi
echo "Completed"

echo "Processing Drive-Library association"
if [ -a $OUTDIR/edc_drives.txt ]; then
	set -A LINES `wc -l $OUTDIR/edc_drives.txt`
	TOP=`expr $LINES - 3`
	END=`expr $TOP - 4`
	head -${TOP} $OUTDIR/edc_drives.txt | tail -${END} > $OUTDIR/edc_drives_processed.txt
	rm $OUTDIR/edc_drives.txt
	while read line ;do
		LIB_NAME=`echo $line | cut -d, -f1`
		echo $line >> "$OUTDIR/edc_drives_${LIB_NAME}.txt"
	done < $OUTDIR/edc_drives_processed.txt
fi
echo "Completed"

echo "Running TSM back end capacity calculation macros"
if [[ $CLA = accurate ]]; then
	if [ -a $OUTDIR/edc_tsmconfig.txt ]; then
		VERSION=`cat $OUTDIR/edc_tsmconfig.txt`
		MAJORVERSION=`echo $VERSION | cut -d, -f1`
		MINORVERSION=`echo $VERSION | cut -d, -f2`
		MACRONAME=none

		if [[ $MAJORVERSION = 6 && $MINORVERSION = 2 ]]; then
			MACRONAME=tsm62.macro
		fi

		if [[ $MAJORVERSION = 6 && $MINORVERSION = 3 ]]; then
			MACRONAME=tsm63.macro
		fi

		if [[ $MAJORVERSION = 7 ]]; then
			MACRONAME=tsm63.macro
		fi

		if [[ $MACRONAME = none ]]; then
			echo "Not running TSM macro commands for version :"$VERSION
		else
			echo "Running TSM macro for computing backend capacity"
			dsmadmc -id=$ID -password=$PASS -server=$SERVER -dataonly=yes -outfile=$OUTDIR/tsmcapacity_report.txt macro ./TSMBackendCapacity/$MACRONAME
		fi

	fi
fi

LogMessage "Running TAR on the output files"
echo "Running TAR command to bundle the output files"

tmstamp=`date +%Y-%m-%d_%H-%M-%S`
tarfilename=edc_data_$tmstamp

tar -cf $OUTDIR/$tarfilename.tar $OUTDIR/
if [[ $? -ne 0 ]]; then 			# check return code passed to function
	echo "TAR command failed to bundle the output files" 	# if rc > 0 then echo error msg and quit
	LogMessage "Failed to tar the output files"
exit $?
fi

echo ""
echo "*********** Please copy $OUTDIR/$tarfilename.tar file. ***************"
